/*-------------------------------------------------------------------
   Name: Xi Dashboard.mq4
   Copyright 2010, Xaphod, http://forexwhiz.appspot.com
   
   Description: Xi Dashboard provides the following info on the chart
        Market Info: 
             Symbol, Bid, Ask and Spread.
        Fractal & Pip info with Up/Down color coding:
              PTFH - Pips to fractal high
	           PTFL - Pips to fractal low
	           PLFS - Pips of Last Fractal Swing.
	           PTO  - Pips to open	   
	           PTMA - Pips to MA
	           BSLF - Bars Since Last Fractal
	      Moving Average:
	           2 MAs for use with PTMA function above, 2 MA Modes
	           0 = Standard/Fixed MA, 1 = Time-frame variable
	      Last H1/H4 high and low with bar up/down color coding	           
	           S/R lines on chart 
	      Daily Range:
	           Calculated from 00:00 to 23:59
	           TDR - Todays Range
	           YDR - Yesterdays Range
	           ADR5 - 5 day ADR. Average Range of the last 5 days
	           ADR20 - 20 day ADR. Average Range of the last 20 days	           
	         
   Change log: 
       2010-03-28. Xaphod, v0.80 
          - First Release 
       2010-03-30. Xaphod, v0.81
          - Fixed: Previous H4 line not being erased.
          - Fixed: PLFS did not change color.
          - Added MA type. Normal and Natural Flow.
          - Made H1/H4 last high/low data and line parameters independent of each other.
          - Added BSLF - Bars Since Last Fractal.
       2010-03-31, Xaphod, v0.82
          - Added dashboard box with optional background color.
          - Improved formatting of data.          
       2010-03-31, Xaphod, v0.83
          -  Release 
          - Fixed pip values off by x100 in 2/3 digit currencies
          - Added dashboard Y-Offset Adjustment parameter
       2010-04-14, Xaphod, v0.84
          - Check for and skip sundays in adr calculation
       2010-05-16, Xaphod, v0.85
          - Added simlation mode
          - Added moving average 2
          - Changed moving average modes. 0=Fixed, 1=Time-frame variable
          - Changed High/Low text color to indicate direction of bar
          - Refactoring
          - Fixed background box resizing on different time-frames
       2010-05-18, Xaphod, v0.86
          - Fixed PTO in SimMode
       To do:
          - Market open time setting
          - Adr based on market open
             
-----------------------------------------------------------------*/
// Indicator properties
#property copyright "Copyright  2010, Xaphod"
#property link      "http://forexwhiz.appspot.com"
#property indicator_chart_window
#property indicator_buffers 6
#property indicator_color1 Salmon
#property indicator_width1 2
#property indicator_color2 Salmon
#property indicator_width2 2
#property indicator_color3 White
#property indicator_width3 2
#property indicator_color4 White
#property indicator_width4 2
#property indicator_color5 Yellow
#property indicator_color6 DeepSkyBlue

#define INDICATOR_NAME "Xi-Dashboard"
#define INDICATOR_VERSION "v0.86"
#define TEXT_XPOS 10
#define TEXT_YINT 13
#define TF_M1 1
#define TF_M5 5
#define TF_M15 15
#define TF_M30 30
#define TF_H1 60
#define TF_H4 240
#define TF_D 1440
#define TF_W 10080
#define TF_M 43200

// Input parameters
extern string   ShowH1HLLinesInfo="<< Show 1H High/Low Lines on Chart >>";
extern bool     ShowH1HLLines=true;
extern string   ShowH1HLDataInfo="<< Show 1H High/Low Data on Dashboard >>";
extern bool     ShowH1HLData=true;
extern string   ShowH4HLLinesInfo="<< Show 4H High/Low Lines on Chart >>";
extern bool     ShowH4HLLines=true;
extern string   ShowH4HLDataInfo="<< Show 4H High/Low Data on Dashboard  >>";
extern bool     ShowH4HLData=true;
extern string   ShowPipsDataInfo="<< Show Pip Range Data on Dashboard >>";
extern bool     ShowPipsData=true;
extern string   TextColorInfo="<< Main Data Text Colors >>";
extern color    UpTextColor=Blue;
extern color    DnTextColor=Maroon;
extern string   BlankLine1="###############################";
extern string   ShowAdrDataInfo="<< Show ADR Data on Dashboard >>";
extern bool     ShowAdrData=true;
extern string   AdrTextColorInfo="<< Adr Text Color >>";
extern color    AdrTextColor=LightBlue;

extern string   BlankLine2="###############################";
extern string   MA1TypeInfo="<<  0=Standard/Fixed MA, 1=TF Variable MA  >>";
extern int      MA1Type=1;
extern int      MA1Period=20; // MA Period. Deactivate: MA=0 
extern string   MA1TFVPeriod="0,12,20,15,12,0,10,12,5";
extern int      MA1Shift=0;
extern int      MA1Mode=0;   // SMA=0, EMA=1, SMMA=2, LWMA=3
extern int      MA1Price=0; // CLOSE=0, OPEN=1, HIGH=2, LOW=3, MEDIAN=4, TYPICAL=5, WEIGHTED=6
extern color    MA1Color=Yellow;
extern string   MA2TypeInfo="<< 0=Standard/Fixed MA, 1=TF Variable MA >>";
extern int      MA2Type=1; 
extern int      MA2Period=240; // MA Standard Period. Deactivate: MA=0 
extern string   MA2TFVPeriod="0,180,240,0,240,0,260,240,0";    // MA TF Variable Period. "1,5,15,60,240,1440,10080,43200";
extern int      MA2Shift=0;
extern int      MA2Mode=0;    // SMA=0, EMA=1, SMMA=2, LWMA=3
extern int      MA2Price=0;   // CLOSE=0, OPEN=1, HIGH=2, LOW=3, MEDIAN=4, TYPICAL=5, WEIGHTED=6
extern color    MA2Color=DodgerBlue;

extern string   BlankLine3="###############################";
extern string   ShowMarketDataInfo="<< Show Market Info in Dashboard >>";
extern bool     ShowMarketData=true;
extern string   MarketTextColorInfo="<< Market Info Text Colors >>";
extern color    SymbolTextColor=Gold;
extern color    BidTextColor=Lime;
extern color    AskTextColor=Maroon;
extern color    SpreadTextColor=Aqua;
extern string   ShowAskInfo="<< Show Ask Data in Dashboard >>";
extern bool     ShowAsk=true;
extern string   ShowSpreadInfo="<< Show Spread Data in Dashboard >>";
extern bool     ShowSpread=true;
extern string   BlankLine4="###############################";
extern string   DisplayCornerInfo="<<Top Left=0, Top Right=1,Bottom left=2, Bottom Right=3>>";
extern int      DisplayCorner=1;
extern color    DashBackgroundColor=DarkGray;
extern int      DisplayYOffset=1;
extern string   BlankLine5="###############################";
extern bool     SimMode=false;

// Indicator buffers
double dExtMapBuffer1[];
double dExtMapBuffer2[];
double dExtMapBuffer3[];
double dExtMapBuffer4[];
double dExtMapBuffer5[];
double dExtMapBuffer6[];

// Module variables
// Dashboard position & size
int iTextYPos = 1;
int iTextCorner = 1;
int iDashHeight=0;
// MA periods
int iMA1Period=0;
int iMA1TFPeriod[9]={0,0,0,0,0,0,0,0,0};
int iMA2Period=0;
int iMA2TFPeriod[9]={0,0,0,0,0,0,0,0,0};

//---- Indicator name 
string sObjId = INDICATOR_NAME;


//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+
int init() {
  // Indicator Settings
  SetIndexStyle(0,DRAW_LINE);
  SetIndexBuffer(0,dExtMapBuffer1);
  SetIndexDrawBegin(0, 60/Period());   
  SetIndexLabel(0,"H1 Last High");
  SetIndexStyle(1,DRAW_LINE);
  SetIndexBuffer(1,dExtMapBuffer2);
  SetIndexDrawBegin(1, 60/Period());
  SetIndexLabel(1,"H1 Last Low");
  SetIndexStyle(2,DRAW_LINE);
  SetIndexBuffer(2,dExtMapBuffer3);
  SetIndexDrawBegin(2, TF_H4/Period()); 
  SetIndexLabel(2,"H4 Last High");  
  SetIndexStyle(3,DRAW_LINE);
  SetIndexBuffer(3,dExtMapBuffer4);
  SetIndexDrawBegin(3, TF_H4/Period());   
  SetIndexLabel(3,"H4 Last Low");
  SetIndexStyle(4,DRAW_LINE,EMPTY, EMPTY, MA1Color);
  SetIndexBuffer(4,dExtMapBuffer5);  
  SetIndexStyle(5,DRAW_LINE,EMPTY, EMPTY, MA2Color);
  SetIndexBuffer(5,dExtMapBuffer6);  
  
  // Set indicator name  
  IndicatorShortName( "[" + INDICATOR_NAME + INDICATOR_VERSION + "]");
  
  // Set lower TF MA Period
  if (MA1Type==0)
    iMA1Period = MA1Period;
  else
    iMA1Period = GetMA1TFV(Period());
  SetIndexLabel(4,"MA"+iMA1Period);
  
  // Set Higher TF MA Period
  if (MA2Type==0)
    iMA2Period = MA2Period;
  else
    iMA2Period = GetMA2TFV(Period());
  SetIndexLabel(5,"MA"+iMA2Period);
  
  // Simulator settings
  if (SimMode==true) 
    ShowAsk=false; 
  
  // Set Constant variables
  iTextCorner = DisplayCorner;  
  iDashHeight=DashboardBoxHeight();
  
  // Exit
  return(0);
}
  
  
//+------------------------------------------------------------------+
//| Custom indicator deinitialization function                       |
//+------------------------------------------------------------------+
int deinit() {
  int i;
  
  // Clear text objects
  for( i=ObjectsTotal()-1; i>-1; i--)
   if (StringFind(ObjectName(i),sObjId)>=0)  ObjectDelete(ObjectName(i));
   
  // Exit
  return(0);
}


//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int start()  {
  //int iNewBars;
  int iNewTick;
  int iCountedBars; 
  int i;  
  
  // Get unprocessed bars
  iCountedBars=IndicatorCounted();
  if(iCountedBars < 0) return (-1); 
  if(iCountedBars>0) iCountedBars--;
  iNewTick=Bars-iCountedBars;
  
  // Display MA
  if (iMA1Period>0)
    for(i=iNewTick-2; i>=0; i--) 
      dExtMapBuffer5[i]=iMA(NULL,0,iMA1Period,MA1Shift,MA1Mode,MA1Price,i);
  if (iMA2Period>0)  
    for(i=iNewTick-2; i>=0; i--) 
      dExtMapBuffer6[i]=iMA(NULL,0,iMA2Period,MA2Shift,MA2Mode,MA2Price,i);
      
  
  //DrawDashBkg();
  if (DashBackgroundColor!=CLR_NONE)
    DrawDashBox();
  
  // Display Market info on Dashboard
  iTextYPos = 1;  
  if (ShowMarketData==true) {
    DisplayMarketInfo(); 
    iTextYPos++; 
  }
  
  // Display fractal & pip data on Dashboard
  if (ShowPipsData==true) {
    FractalPipData();
    PipsToOpen();
    PipsToMA();
    iTextYPos++;
  }
  
  // Display H1 and H4 last high/low on Dashboard and draw lines
  SetH1LastHigh();
  SetH4LastHigh();
  if ((ShowH1HLData==true || ShowH4HLData==true) && Period()<TF_H4)
    iTextYPos++;  
    
  // Display daily range & pip data on Dashboard
  if (ShowAdrData==true) 
    DailyRange();
    
  // exit
  return(0);
}
//+------------------------------------------------------------------+


//-----------------------------------------------------------------------------
// function: GetMA1TFV()
// Description: Get MA1 period based on the current time-frame & MA1TFVPeriod
//-----------------------------------------------------------------------------
int GetMA1TFV(int iPeriod) {
   string sSubStr=MA1TFVPeriod;
   int i,iTokenEnd;
   
   for (i=0;i<9;i++) {
     iTokenEnd=StringFind(sSubStr,",",0);
     if (iTokenEnd<0) {
       iMA1TFPeriod[i]=StrToInteger(sSubStr);
       break;
     }
     else {       
       iMA1TFPeriod[i]=StrToInteger(StringSubstr(sSubStr,0,iTokenEnd));
       sSubStr=StringSubstr(sSubStr,iTokenEnd+1,StringLen(sSubStr));       
     }  
   }
   
   switch(iPeriod) {
    case TF_M1: return(iMA1TFPeriod[0]);
    case TF_M5: return(iMA1TFPeriod[1]);
    case TF_M15: return(iMA1TFPeriod[2]);
    case TF_M30: return(iMA1TFPeriod[3]);
    case TF_H1: return(iMA1TFPeriod[4]);
    case TF_H4: return(iMA1TFPeriod[5]);
    case TF_D: return(iMA1TFPeriod[6]);
    case TF_W: return(iMA1TFPeriod[7]);
    case TF_M: return(iMA1TFPeriod[8]);
    default: return(0);
  }
  return(0);
}

//-----------------------------------------------------------------------------
// function: GetMA2TFV()
// Description: Get MA2 period based on the current time-frame & MA2TFVPeriod
//-----------------------------------------------------------------------------
int GetMA2TFV(int iPeriod) {
   string sSubStr=MA2TFVPeriod;
   int i,iTokenEnd;
   
   for (i=0;i<9;i++) {
     iTokenEnd=StringFind(sSubStr,",",0);
     if (iTokenEnd<0) {
       iMA2TFPeriod[i]=StrToInteger(sSubStr);
       break;
     }
     else {       
       iMA2TFPeriod[i]=StrToInteger(StringSubstr(sSubStr,0,iTokenEnd));
       sSubStr=StringSubstr(sSubStr,iTokenEnd+1,StringLen(sSubStr));       
     }  
   }
  
  switch(iPeriod) {
    case TF_M1: return(iMA2TFPeriod[0]);
    case TF_M5: return(iMA2TFPeriod[1]);
    case TF_M15: return(iMA2TFPeriod[2]);
    case TF_M30: return(iMA2TFPeriod[3]);
    case TF_H1: return(iMA2TFPeriod[4]);
    case TF_H4: return(iMA2TFPeriod[5]);
    case TF_D: return(iMA2TFPeriod[6]);
    case TF_W: return(iMA2TFPeriod[7]);
    case TF_M: return(iMA2TFPeriod[8]);
    default: return(0);
  }
  return(0); 
}


//-----------------------------------------------------------------------------
// function: SetH1LastHigh()
// Description: Calc and display H1 High and Low 
//-----------------------------------------------------------------------------
void SetH1LastHigh() {
  double dHigh;
  double dLow;
 
  double dBid;
  int iNrOfBars;
  int iPeriod;
  int iShift=-1;
  int i;
  color cTextColor;
  int iY;
  
  // Get and display 14 high & low and draw SR lines
  iPeriod=GetPeriod(Period());  
  if (iPeriod<TF_H1) {  
    // Get index of the end of the last 1H bar
    for (iShift=0; iShift<Bars; iShift++) 
      if (TimeMinute(Time[iShift])== 0) 
        break;
    iShift++;    
    
    // Get nr of bars for the 1H period. There can be bars missing!
    for (i=iShift; i<Bars; i++) 
      if (TimeMinute(Time[i])== 0) 
        break;
    iNrOfBars=i-iShift+1;   
    
    // Find the highest and lowest data for specified nr of bars
    dHigh=High[iHighest(NULL,0,MODE_HIGH,iNrOfBars,iShift)];
    dLow=Low[iLowest(NULL,0,MODE_LOW,iNrOfBars,iShift)];
    
    // Get bid value
    if (SimMode==true) 
      dBid=Close[0]; 
    else 
      dBid=Bid;
  
    // Display on dashboard    
    if (ShowH1HLData==true) {      
      if (Open[iNrOfBars+iShift]>Close[iShift]) cTextColor=DnTextColor; else cTextColor=UpTextColor;
      iY=TEXT_YINT*iTextYPos+DisplayYOffset;
      WriteText("LH(H1)  " + DoubleToStr(dHigh, Digits), sObjId+"_LH_H1", iTextCorner, TEXT_XPOS, iY,cTextColor);
      iTextYPos++;
      iY=TEXT_YINT*iTextYPos+DisplayYOffset;
      WriteText("LL(H1)  " + DoubleToStr(dLow, Digits), sObjId+"_LL_H1", iTextCorner, TEXT_XPOS, iY,cTextColor);
      iTextYPos++;
    }
        
    // Draw H/L line
    if (ShowH1HLLines==True)  {
      // Clear old lines
      for(i=iNrOfBars+iShift; i<iNrOfBars+iShift+(1200/iPeriod)+2; i++) {    
        dExtMapBuffer1[i]=EMPTY_VALUE;
        dExtMapBuffer2[i]=EMPTY_VALUE;    
      } 
      // Draw new lines
      for(i=iShift; i<iNrOfBars+iShift; i++) {    
        dExtMapBuffer1[i]=dHigh;
        dExtMapBuffer2[i]=dLow;    
      }   
    } // end if 
  } // endif
} // end funcion()


//-----------------------------------------------------------------------------
// function: SetH4LastHigh()
// Description: Calc and display H4 High and Low 
//-----------------------------------------------------------------------------
void SetH4LastHigh() {
  double dHigh;
  double dLow;
  double dBid;
  int iNrOfBars;
  int iPeriod;
  int iShift;
  int i;
  color cTextColor;
  int iY;
  
  // Get and display H4 high & low and draw SR lines
  iPeriod=GetPeriod(Period());  
  if (iPeriod<TF_H4) {
  
    // Get index of the end of the last 4H bar
    for (iShift=0; iShift<Bars; iShift++)
      if (TimeMinute(Time[iShift])== 0 && MathRound(MathMod(TimeHour(Time[iShift]),4))==0) 
        break;
    iShift++;
    
    // Get nr of bars for the 4H period. There can be bars missing!
    for (i=iShift; i<Bars; i++) 
      if (TimeMinute(Time[i])== 0 && MathRound(MathMod(TimeHour(Time[i]),4))==0) 
        break;
    iNrOfBars=i-iShift+1;
   
    // Calc Hih and low for specified nr of bars
    dHigh=High[iHighest(NULL,0,MODE_HIGH,iNrOfBars,iShift)];
    dLow=Low[iLowest(NULL,0,MODE_LOW,iNrOfBars,iShift)];  
    
    // Get bid value
    if (SimMode==true) 
      dBid=Close[0]; 
    else 
      dBid=Bid;
    
    // Display on dashboard
    if (ShowH4HLData==true) {      
      if (Open[iNrOfBars+iShift]>Close[iShift]) cTextColor=DnTextColor; else cTextColor=UpTextColor ;
      iY=TEXT_YINT*iTextYPos+DisplayYOffset;
      WriteText("LH(H4)  " + DoubleToStr(dHigh, Digits), sObjId+"_LH-H4", iTextCorner, TEXT_XPOS, iY,cTextColor);
      iTextYPos++;   
      iY=TEXT_YINT*iTextYPos+DisplayYOffset;
      WriteText("LL(H4)  " + DoubleToStr(dLow, Digits), sObjId+"_LL-H4", iTextCorner, TEXT_XPOS, iY,cTextColor);    
      iTextYPos++;
    }  
    
    // Draw H/L line 
    if (ShowH4HLLines==True)  {
      // Clear old lines
      for(i=iNrOfBars+iShift; i<iNrOfBars+iShift+(480/iPeriod)+2; i++) {    
        dExtMapBuffer3[i]=EMPTY_VALUE;
        dExtMapBuffer4[i]=EMPTY_VALUE; 
      } 
      // Draw new lines
      for(i=iShift; i<iNrOfBars+iShift; i++) {    
        dExtMapBuffer3[i]=dHigh;
        dExtMapBuffer4[i]=dLow;   
      } 
      
    } // endif
  } // endif
} //end funcion()


//-----------------------------------------------------------------------------
// function: FractalPipData()
// Description: Calc and display:
//        PLFS - Pips of Last Fractal Swing, 
//        PTFH - Pips To Fractal High, 
//        PTFL - Pips To Fractal Low 
//        PTO - Pips to open
//        PTMA - Pips to MA
//-----------------------------------------------------------------------------
void FractalPipData() {
  double dBid;
  double dFractalHigh;
  int iFractalHigh;
  double dPTFH;
  double dFractalLow;
  int iFractalLow;
  double dPTFL;
  double dFractalSwing;
  int iBSLF;
  color cTextColor;
  int iY;
  
  // Get bid value
  if (SimMode==true) 
    dBid=Close[0]; 
  else 
    dBid=Bid;
  
  // Calc Fractals - PLFS - Pips of Last Fractal Swing, PTFH - Pips To Fractal High, PTFL - Pips To Fractal Low 
  iFractalHigh=GetLastFractalIndex( 0, MODE_UPPER);
  iFractalLow=GetLastFractalIndex( 0, MODE_LOWER);    
  
  if (iFractalHigh>=0) 
    dPTFH=PriceToPips( dBid - High[iFractalHigh]);
  if (iFractalLow>=0) 
    dPTFL=PriceToPips( dBid - Low[iFractalLow]);
  dFractalSwing=PriceToPips(High[iFractalHigh]-Low[iFractalLow]); 
  
  // Display Fractal data
  if (dPTFH<=0) cTextColor=DnTextColor; else cTextColor=UpTextColor ;
  iY=TEXT_YINT*iTextYPos+DisplayYOffset;
  WriteText(" PTFH  " + DoubleToStr(MathAbs(dPTFH), PipPrecision()), sObjId+"_PTFH", iTextCorner, TEXT_XPOS, iY,cTextColor);
  iTextYPos++;
  if (dPTFL<=0) cTextColor=DnTextColor; else cTextColor=UpTextColor ;
  iY=TEXT_YINT*iTextYPos+DisplayYOffset;
  WriteText(" PTFL  " + DoubleToStr(MathAbs(dPTFL), PipPrecision()), sObjId+"_PTFL", iTextCorner, TEXT_XPOS, iY,cTextColor);
  iTextYPos++;
  
  if (iFractalHigh>iFractalLow) cTextColor=DnTextColor; else cTextColor=UpTextColor ;
  iY=TEXT_YINT*iTextYPos+DisplayYOffset;
  WriteText(" PLFS  " + DoubleToStr(dFractalSwing, PipPrecision()), sObjId+"_PLFS", iTextCorner, TEXT_XPOS, iY,cTextColor);
  iTextYPos++;
  
  // Calc BSLF - Bars Since Last Fractal
  if (iFractalLow>=iFractalHigh) {
    iBSLF=iFractalHigh;
    if (dBid > High[iFractalHigh]) cTextColor=UpTextColor; else cTextColor=DnTextColor;
  }
  else {
    iBSLF=iFractalLow;
    if (dBid > Low[iFractalLow]) cTextColor=UpTextColor; else cTextColor=DnTextColor;
  }
  iY=TEXT_YINT*iTextYPos+DisplayYOffset;
  WriteText(" BSLF  " + iBSLF,sObjId+"_BSLF", iTextCorner, TEXT_XPOS, iY,cTextColor);
  iTextYPos++;
} // end funcion()


//-----------------------------------------------------------------------------
// function: PipsToOpen()
// Description: Calc and display:
//        PTO - Pips to open
//-----------------------------------------------------------------------------
void PipsToOpen() {
  double dBid;
  double dPTO;
  color cTextColor;
  int iY;
  datetime tOpenTime;
  
  // Get bid value
  if (SimMode==true) 
    dBid=Close[0]; 
  else 
    dBid=Bid;
    
  // Calc PTO - Pips to open
  tOpenTime=StrToTime(TimeToStr(Time[0],TIME_DATE)+" "+"00:00");
  if (SimMode==true) 
    dPTO=PriceToPips( dBid-iOpen(NULL,0,iBarShift(Symbol(), 0, tOpenTime)));
  else 
    dPTO=PriceToPips( dBid-iOpen(NULL,PERIOD_D1,0));
  if (dPTO<=0) cTextColor=DnTextColor; else cTextColor=UpTextColor ;
  iY=TEXT_YINT*iTextYPos+DisplayYOffset;
  WriteText("  PTO  " + DoubleToStr( MathAbs(dPTO) ,PipPrecision()),sObjId+"_PTO", iTextCorner, TEXT_XPOS, iY,cTextColor);
  iTextYPos++;
  
}


//-----------------------------------------------------------------------------
// function: PipsToMA()
// Description: Calc and display:
//        PTMA - Pips to MA
//-----------------------------------------------------------------------------
void PipsToMA() {
  double dBid;
  double dPTMA;
  double dCurMA;
  color cTextColor;
  int iY;
  
  // Get bid value
  if (SimMode==true) 
    dBid=Close[0]; 
  else 
    dBid=Bid;
    
  // Calc PTMA - Pips to MA
  if (iMA1Period>0) {
    dCurMA=iMA(NULL,0,iMA1Period,MA1Shift,MA1Mode,MA1Price,0);
    dPTMA=PriceToPips(dBid-dCurMA);
    if (dBid<dCurMA) cTextColor=DnTextColor; else cTextColor=UpTextColor ;
    iY=TEXT_YINT*iTextYPos+DisplayYOffset;
    WriteText("PTMA1  " + DoubleToStr(MathAbs(dPTMA) , PipPrecision()), sObjId+"_PTMA1", iTextCorner, TEXT_XPOS, iY,cTextColor);
    iTextYPos++;
  }
    
  if (iMA2Period>0) {
    dCurMA=iMA(NULL,0,iMA2Period,MA2Shift,MA2Mode,MA2Price,0);
    dPTMA=PriceToPips(dBid-dCurMA);
    if (dBid<dCurMA) cTextColor=DnTextColor; else cTextColor=UpTextColor ;
    iY=TEXT_YINT*iTextYPos+DisplayYOffset;
    WriteText("PTMA2  " + DoubleToStr(MathAbs(dPTMA) , PipPrecision()), sObjId+"_PTMA2", iTextCorner, TEXT_XPOS, iY,cTextColor);
    iTextYPos++;    
  }  
}


//-----------------------------------------------------------------------------
// function: GetLastFractalIndex()
// Description: Get the index of the latest High or Low Fractal
//-----------------------------------------------------------------------------
int GetLastFractalIndex(int iTimeFrame, int iMode) {
  
  for (int i=0; i<Bars; i++) 
    if (iFractals(NULL, iTimeFrame, iMode, i)!=NULL) return(i); 

  return(-1);
}  // end funcion()


//-----------------------------------------------------------------------------
// function: DailyRange()
// Description: Calc and display the daily range
//-----------------------------------------------------------------------------
void DailyRange() {
  double dTDR;
  static double dYDR;
  static double d5DR;
  static double d20DR;
  static datetime dDay=0;
  int iY;

  // Calc the ADR
  if (SimMode==true) {
   dTDR=PriceToPips(CalcTDR());
   if (dDay!=TimeDay(Time[0])) {
     dDay=TimeDay(Time[0]);
     dYDR=PriceToPips(CalcADR(1));
     d5DR=PriceToPips(CalcADR(5));
     d20DR=PriceToPips(CalcADR(20));
    }
  }
  else {  
    dTDR=PriceToPips(iHigh(NULL,PERIOD_D1,0)-iLow(NULL,PERIOD_D1,0));
    if (dDay!=TimeDay(Time[0])) {
      dDay=TimeDay(Time[0]);
      dYDR=PriceToPips(CalcDailyRange(1));
      d5DR=PriceToPips(CalcDailyRange(5));
      d20DR=PriceToPips(CalcDailyRange(20));
    }
  }
  
  // Display ADR on dashboard
  iY=TEXT_YINT*iTextYPos+DisplayYOffset;
  WriteText("  TDR  " + DoubleToStr(dTDR, PipPrecision())+"  ", sObjId+"_TDR", iTextCorner, TEXT_XPOS, iY, AdrTextColor);
  iTextYPos++;
  iY=TEXT_YINT*iTextYPos+DisplayYOffset;
  WriteText("  YDR  " + DoubleToStr(dYDR, PipPrecision())+"  ", sObjId+"_YDR", iTextCorner, TEXT_XPOS, iY, AdrTextColor);
  iTextYPos++;
  iY=TEXT_YINT*iTextYPos+DisplayYOffset;
  WriteText(" ADR5  " + DoubleToStr(d5DR, PipPrecision())+"  ", sObjId+"_ADR5", iTextCorner, TEXT_XPOS, iY, AdrTextColor);
  iTextYPos++;
  iY=TEXT_YINT*iTextYPos+DisplayYOffset;
  WriteText("ADR20  " + DoubleToStr(d20DR, PipPrecision())+"  ", sObjId+"_ADR20", iTextCorner, TEXT_XPOS, iY, AdrTextColor);
  iTextYPos++;
}  // end funcion()


//-----------------------------------------------------------------------------
// function: CalcDailyRange()
// Description: Calc the average daily range over a number of days
//-----------------------------------------------------------------------------
double CalcDailyRange(int iNrOfDays) {
  double dADR=0;
  int i;
  int j=1;
 
  for (i=1; i<=iNrOfDays; i++) {
    while (TimeDayOfWeek(iTime(NULL,PERIOD_D1,j))==0) j++; // Skip sundays
    dADR=dADR+ iHigh(NULL,PERIOD_D1,j)-iLow(NULL,PERIOD_D1,j);
    j++;
  }  
 
  return (dADR/iNrOfDays);  
} //end funcion()


//-----------------------------------------------------------------------------
// function: CalcTDR()
// Description: Calc todays daily range. For sim
//-----------------------------------------------------------------------------
double CalcTDR() {
  int i;
  int iShift=0;
  int iNrOfBars=0;
  double dHigh;
  double dLow;

  // Get nr of bars for the 4H period. There can be bars missing!
  for (i=0; i<Bars; i++) 
    if (TimeHour(Time[i])==0 && TimeMinute(Time[i])==0)
      break;
       
  // Calc Hih and low for specified nr of bars
  dHigh=High[iHighest(NULL,0,MODE_HIGH,i+1,0)];
  dLow=Low[iLowest(NULL,0,MODE_LOW,i+1,0)];  
    
  return(dHigh-dLow);
}


//-----------------------------------------------------------------------------
// function: CalcTDR()
// Description: Calc todays daily range. For sim
//-----------------------------------------------------------------------------
double CalcADR(int iNrOfDays) {
  int i,j;
  int iShiftBeg=0;
  int iShiftEnd=0;
  int iNrOfBars=0;
  double dHigh;
  double dLow;
  double dADR=0;

  // Get nr of bars form the last 00:000 hours
  for (i=0; i<Bars; i++) 
    if (TimeHour(Time[i])==0 && TimeMinute(Time[i])==0)
      break;
  
  // start from end of day, 23:59
  iShiftBeg=i+1;
  for (i=1; i<=iNrOfDays; i++) {    
    for (j=iShiftBeg; j<Bars; j++) {
      if (TimeHour(Time[j])==0 && TimeMinute(Time[j])==0)
        break;
    }
    iNrOfBars=j-iShiftBeg;    
    // Calc Hih and low for specified nr of bars    
    if (iNrOfBars>(12*(60.0/Period()))) { // Check that we have at least half a day of bars (no sundays!)
      dHigh=High[iHighest(NULL,0,MODE_HIGH,iNrOfBars,iShiftBeg)];
      dLow=Low[iLowest(NULL,0,MODE_LOW,iNrOfBars,iShiftBeg)];  
      dADR=dADR+ (dHigh-dLow);
    }
    iShiftBeg=iShiftBeg+iNrOfBars+1;
  }
  return(dADR/(i-1));
}


//-----------------------------------------------------------------------------
// function: DisplayMarketInfo()
// Description: Calc the average daily range over a number of days
//-----------------------------------------------------------------------------
void DisplayMarketInfo() {
  int iX,iY;
  double dBid;
  
  // Get bid value
  if (SimMode==true) 
    dBid=Close[0]; 
  else 
    dBid=Bid;
  
  // Instrument  
  if (DisplayCorner==1 || DisplayCorner==3) iX=90; else iX=10;  // Set display positions depending the corner
  if (DisplayCorner==2 || DisplayCorner==3) iY=TEXT_YINT*iTextYPos+14+DisplayYOffset; else iY=TEXT_YINT*iTextYPos-3+DisplayYOffset;
  WriteText(GetShortSymbol(Symbol()), sObjId+"_Symbol", iTextCorner, iX, iY,SymbolTextColor,24);
  //iTextYPos++;
  
  // Price
  if (DisplayCorner==0 || DisplayCorner==2) iX=50; else iX=10; // Set display positions depending the corner
  iY=TEXT_YINT*iTextYPos+DisplayYOffset;
  WriteText(DoubleToStr( dBid, Digits), sObjId+"_Bid", iTextCorner, iX, iY,BidTextColor, 14);
  iTextYPos++;  
  
  if (ShowAsk==True) {
    if (DisplayCorner==0 || DisplayCorner==2) iX=52; else iX=35; // Set display positions depending the corner
    iY=TEXT_YINT*iTextYPos+DisplayYOffset;
    WriteText( DoubleToStr(Ask, Digits), sObjId+"_Ask", iTextCorner, iX, iY,AskTextColor,9);
    iTextYPos++;
  }
  
  if (ShowSpread==True) {
    if (ShowAsk==False) 
      if (DisplayCorner==0 || DisplayCorner==2) iX=50; else iX=35;
    else
      if (DisplayCorner==0 || DisplayCorner==2) iX=50; else iX=33;
    iY=TEXT_YINT*iTextYPos+DisplayYOffset;  
    WriteText("SP  "+ DoubleToStr( PriceToPips(Ask-Bid) , PipPrecision()), sObjId+"_Spread", iTextCorner, iX, iY,SpreadTextColor,9);      
  iTextYPos++;
  }
  
}  // end funcion()
 
 
//-----------------------------------------------------------------------------
// function: WriteText()
// Description: Write text strings to window
//-----------------------------------------------------------------------------
void WriteText(string sText, string sTextId, int iCorner, int iXPos, int iYPos, color cTxtColor=LightBlue, int iTxtSize=9) {
  int i;      
  int iMaxStrLen=14+PipPrecision();
  int iBoxHeight=(iDashHeight)*TEXT_YINT;

  // Delete text object if it already exists      
  if (ObjectFind(sTextId)>0 ) ObjectDelete(sTextId); 
  
  // Adjust text positioning
  if ((iCorner==1 || iCorner==3) ) {
    if (StringLen(sText)<iMaxStrLen && (iYPos-DisplayYOffset)/TEXT_YINT>3) sText=sText+PadStr(iMaxStrLen-StringLen(sText)-1); // Pad on right side
  }
  else {
    if (StringLen(sText)<iMaxStrLen && (iYPos-DisplayYOffset)/TEXT_YINT>3) sText=" "+sText; // Pad on left side
  }
  
  if (iCorner==2 || iCorner==3) {
    iYPos=iBoxHeight+DisplayYOffset-iYPos;    // Flip coordinates if corner=bottom
    iYPos=DisplayYOffset+iYPos;
  }
  
  // Print Text
  ObjectCreate(sTextId, OBJ_LABEL, 0, 0, 0, 0, 0);
  ObjectSet(sTextId, OBJPROP_CORNER, iCorner);
  ObjectSet(sTextId, OBJPROP_WIDTH, 3);
  ObjectSet(sTextId, OBJPROP_XDISTANCE, iXPos);
  ObjectSet(sTextId, OBJPROP_YDISTANCE, iYPos);
  ObjectSet(sTextId, OBJPROP_BACK, false);
  ObjectSetText(sTextId, sText, iTxtSize, "Courier New", cTxtColor);
}  // end funcion()


//-----------------------------------------------------------------------------
// function: DrawDashBox()
// Description: Draw the dashboard box
//-----------------------------------------------------------------------------
void DrawDashBox() {
  string sTextId;
  int iTextSize=10;
  int iXOffset=5;
  int iYOffset=DisplayYOffset;
  
  for (int i=0; i<=iDashHeight; i++) {  
    sTextId=sObjId+"_1BoxLine_"+i;    
    if (ObjectFind(sTextId)>0 ) ObjectDelete(sTextId);    
    ObjectCreate(sTextId, OBJ_LABEL, 0, 0, 0, 0, 0);
    ObjectSet(sTextId, OBJPROP_CORNER, iTextCorner);
    ObjectSet(sTextId, OBJPROP_WIDTH, 3);
    ObjectSet(sTextId, OBJPROP_XDISTANCE, TEXT_XPOS-iXOffset);
    ObjectSet(sTextId, OBJPROP_YDISTANCE, TEXT_YINT*i+iYOffset);
    ObjectSet(sTextId, OBJPROP_BACK, false);
    ObjectSetText(sTextId, "gggggggggg", iTextSize, "Webdings", DashBackgroundColor);  
    } // end for
} // end funcion()


//-----------------------------------------------------------------------------
// function: DashboardBoxHeight()
// Description: Calculate the height of the dashboard box
//-----------------------------------------------------------------------------
int DashboardBoxHeight() {
  int iSize=0;
  
  // Market Info
  if (ShowMarketData==true) {    
    if (ShowSpread==true && ShowAsk==true)
      iSize=iSize+4;
    else if (ShowSpread==false && ShowAsk==true)
      iSize=iSize+3;
    else if (ShowSpread==true && ShowAsk==false)
      iSize=iSize+3;  
    else
      iSize=iSize+2;
  }  

  // Fractal & Pip data
  if (ShowPipsData==true && iMA1Period>0 && iMA2Period>0)
      iSize=iSize+8; 
  else if (ShowPipsData==true && (iMA1Period>0 || iMA2Period>0))
      iSize=iSize+7;
  else if (ShowPipsData==true)
      iSize=iSize+6;
  
  // H1 & H4 high and low data
  if (ShowH1HLData==true && ShowH4HLData==true && Period()<TF_H1)
    iSize=iSize+5;
  else if (ShowH1HLData==true && ShowH4HLData==true && Period()<TF_H4)    
    iSize=iSize+3;
  else if (ShowH1HLData==true && ShowH4HLData==false)    
    iSize=iSize+3;
  else if (ShowH1HLData==false && ShowH4HLData==true)    
    iSize=iSize+3;
  
  // ADR data
  if (ShowAdrData==true)
    iSize=iSize+4;
  
  if (iSize>4)
    iSize=iSize+1;

  return(iSize);
} // end funcion()


//-----------------------------------------------------------------------------
// function: GetPeriod()
// Description: Get the current period. Translate vHandsTrade simulator periods
//              to standard  periods.
//-----------------------------------------------------------------------------
int GetPeriod(int iCurPeriod)  {
  int iResult;
  
  if (iCurPeriod==0) iCurPeriod=Period();
  
  switch (iCurPeriod) {
    case 2: iResult=TF_M1; break;   
    case 6: iResult=TF_M5; break;   
    case 16: iResult=TF_M15; break;
    case 31: iResult=TF_M30; break;
    case 61: iResult=TF_H1; break;
    case 241: iResult=TF_H4; break;
    case 1441: iResult=TF_D; break;
    case 10081: iResult=TF_W; break;
    case 43201: iResult=TF_M; break;
    default: iResult=iCurPeriod; 
  }
  return(iResult);
} // end funcion()


//-----------------------------------------------------------------------------
// function: PriceToPips()
// Description: Convert a proce difference to pips.
//-----------------------------------------------------------------------------
double PriceToPips(double dPrice) {

  if (Digits==2 || Digits==3) 
    return(dPrice/0.01); 
  else if (Digits==4 || Digits==5) 
    return(dPrice/0.0001); 
  else
    return(dPrice);            
} // end funcion()


//-----------------------------------------------------------------------------
// function: PipPrecision()
// Description: Digits of the pips
//-----------------------------------------------------------------------------
double PipPrecision() {

 if (Digits==3 || Digits==5) 
    return(1); 
  else if (Digits==2 || Digits==4) 
    return(0); 
  else
    return(0);            
} // end funcion()
  

//-----------------------------------------------------------------------------
// function: GetShortSymbol()
// Description: Shorten a symbol name. Tex. EURUSD = EU, GBPUSD = GU
//-----------------------------------------------------------------------------
string GetShortSymbol(string sSymbol) {
  string sResult="";
  
  if (StringLen(sSymbol)>5)
    sResult=StringSubstr( sSymbol, 0, 1)+ StringSubstr( sSymbol, 3, 1);
  else
    sResult="sSymbol";
 
  return(sResult);
} // end funcion()

//-----------------------------------------------------------------------------
// function: PadStr()
// Description: Create a blank string N chars long
//-----------------------------------------------------------------------------
string PadStr(int iSize) {
  string sReturn ="";
  
  for (int i=0; i<iSize; i++)
    sReturn=sReturn+" ";
  return(sReturn);
} // end funcion()


